function [lambda xmin L gof] = expFit( x )
%EXPFIT Perform exponential fit of data in x
%   EXPFIT(X) Will fit the data in x to an expontial distribution, in which
%   p(x) ~ exp(-lambda*x)
%
%   This uses an approach modified from:
%   Clauset,Newman,Shalizi Power law Distributions in Empirical Data, 2007
%
%   Only implemented for x integer-valued (did not implement for
%   continuous-valued x)

% reshape x into a column vector
x = reshape(x,length(x),1);

if ~isempty(setdiff(x,floor(x)))
    error('x must be integer-valued')
end

% try every value in x as a potential xmin
xmins = unique(x);
z = x;
xmax = max(z);
dat = zeros(length(xmins),2);
for i = 1:length(xmins)
    xmin = xmins(i);
    %calculate the MLE for lambda for this value of xmin
    z2 = z(z>=xmin);
    N = length(z2);
    lambda = log((N + sum(z2) - N*xmin)/(sum(z2) - N * xmin));
    
    % calculate the CDF for x (the cumulative distr function)
    cdf = cumsum(hist(z2,(xmin:xmax))./N)';
    % calculate the CDF for the fitted function
    fcdf = 1 - exp(-lambda.*((xmin:xmax)' - xmin));
    % calculate the Kolmogorov-Smirnov statistic for this value of xmin
    D = max(abs(cdf-fcdf));
    dat(i,:) = [D,lambda];
end

% Pick the xmin with the smallest KS statistic
[gof idx] = min(dat(:,1));
xmin = xmins(idx);
lambda = dat(idx,2);

z     = x(x>=xmin);
N     = length(z);

L = N*log(1-exp(-lambda)) + N*lambda*xmin  - lambda*sum(z);    
    


end

